--- Base page layout class.
--
-- @license MIT
-- @copyright (c) 2022-2025 Omikhkeia / Didier Willis
-- @module resilient.layouts.base

--- Base layout class.
-- @type resilient.layouts.base

local layout = pl.class()

--- (Constructor) Create a new layout instance.
-- @tparam table _ Options (not used here)
function layout:_init (_)
  self.inner = "width(page)/6"
  self.outer = "width(page)/6"
  self.head = "height(page)/6"
  self.foot = "height(page)/6"
  self.offset = "0"
end

--- Set the paper size.
--
-- This is sort of a hack, as some layouts are hard to implement with SILE's frame
-- specification model (and Cassowary constraints).
--
-- @tparam SILE.types.measurement W Paper width
-- @tparam SILE.types.measurement H Paper height
function layout:setPaperHack (W, H)
  -- unused here but see Canonical layout
  self.W = W
  self.H = H
end

--- Set the binding offset.
--
-- @tparam SILE.types.measurement offset Binding offset
function layout:setOffset (offset)
  self.offset = offset
end

--- Compute the frameset for this layout.
--
-- @treturn table odd Frameset for odd pages
-- @treturn table even Frameset for even pages
function layout:frameset ()
  local odd = {
    textblock = self:textblock(true),
    content = self:content(true),
    footnotes = self:footnotes(true),
    footer = self:footer(true),
    header = self:header(true),
    margins = self:margins(true),
    bindinggutter = self:gutter(true),
  }
  local even = {
    textblock = self:textblock(false),
    content = self:content(false),
    footnotes = self:footnotes(false),
    footer = self:footer(false),
    header = self:header(false),
    margins = self:margins(false),
    bindinggutter = self:gutter(false),
  }
  odd.folio = pl.tablex.copy(odd.footer)
  even.folio = pl.tablex.copy(even.footer)
  -- N.B. At some point we may want folio in headers, but
  -- we need to a way to "link" nofoliothispage with noheaderthispage
  -- in that case!
  return odd, even
end

--- Return the textblock frame specification.
-- @tparam boolean isOdd Whether this is for an odd page
-- @treturn table Frame specification
function layout:textblock (isOdd)
  local left = isOdd and (self.inner .. " + " .. self.offset) or (self.outer .. " - " .. self.offset)
  local right = isOdd and (self.outer  .. " - " .. self.offset) or (self.inner  .. " + " .. self.offset)
  return {
    left = "left(page) + " .. left,
    right = "right(page) - " .. right,
    top = "top(page) + " .. self.head,
    bottom = "bottom(page) - " .. self.foot
  }
end

--- Return the content frame specification.
-- @tparam boolean _ Whether this is for an odd page (unused here)
-- @treturn table Frame specification
function layout:content (_)
  return {
    left = "left(textblock)",
    right = "right(textblock)",
    top = "top(textblock)",
    bottom = "top(footnotes)"
  }
end

--- Return the footnotes frame specification.
-- @tparam boolean _ Whether this is for an odd page (unused here)
-- @treturn table Frame specification
function layout:footnotes (_)
  return {
    left = "left(textblock)",
    right = "right(textblock)",
    height = "0",
    bottom = "bottom(textblock)"
   }
end

-- None of the resources I consulted (Lacroux, Bringhurst, Tschichold...)
-- explain where bottom footer and headers should go, beyond generalities
-- (e.g. "close to the text block").
-- Looking at several books I own, whatever their quality (or lack of
-- thereof), none seem to show any explicit rule in that matter.
-- Hence, I decided to be a typographer on my own:
--  - Let's reserve up to 16pt for these.
--  - Let's play with a golden ratio in that approximate mix.
-- And guess what, it looks "decent" for most standard layouts and page
-- dimensions I checked. According to my tastes, at least.

--- Return the footer frame specification.
-- @tparam boolean _ Whether this is for an odd page (unused here)
-- @treturn table Frame specification
function layout:footer (_)
  return {
    left = "left(textblock)",
    right = "right(textblock)",
    top = "bottom(page) - (" .. self.foot .. ") / 1.618 + 8pt",
    bottom = "top(footer) + 16pt"
  }
end

--- Return the header frame specification.
-- @tparam boolean _ Whether this is for an odd page (unused here)
-- @treturn table Frame specification
function layout:header (_)
  return {
    left = "left(textblock)",
    right = "right(textblock)",
    top = "top(page) + (" .. self.head  .. ") / 1.618 - 8pt",
    bottom = "top(header) + 16pt"
  }
end

--- Return the margins frame specification.
-- @tparam boolean odd Whether this is for an odd page
-- @treturn table Frame specification
function layout:margins (odd)
  return {
    left = odd and "right(textblock) + 2.5%pw" or "left(page) + 0.5in",
    right= odd and "right(page) - 0.5in" or "left(textblock) - 2.5%pw",
    top = "top(textblock)",
    bottom = "bottom(textblock)",
  }
end

--- Return the binding gutter frame specification.
-- @tparam boolean isOdd Whether this is for an odd page
-- @treturn table Frame specification
function layout:gutter (isOdd)
  return {
    left = isOdd and ("left(page) + " .. self.offset) or "left(page)",
    right = isOdd and "left(page)" or ("left(page) + " .. self.offset),
    top = "top(page)",
    bottom = "bottom(page)"
  }
end

-- layout graph drawing adapter

local PathRenderer = require("grail.renderer")
local RoughPainter = require("grail.painters.rough")

local function buildFrameRect (painter, frame, wratio, hratio, options)
  options = options or {}
  local path = painter:rectangle(
    frame:left():tonumber() * wratio,
    frame:top():tonumber() * hratio,
    frame:width():tonumber() * wratio,
    frame:height():tonumber() * hratio, {
      fill = options.fillcolor and SILE.types.color(options.fillcolor) or "none", -- FIXME
      -- Actually the rough option does not work here, and will crash on the "none" fill:
      -- Fill discrepancy between rough and non-rough painter, see https://github.com/Omikhleia/grail/issues/1
      stroke = options.strokecolor and SILE.types.color(options.strokecolor),
      preserveVertices = true,
      disableMultiStroke = true,
      strokeWidth = SU.cast("measurement", options.stroke or "0.3pt"):tonumber(),
      -- Default hachure gets problematic on very thin areas (e.g. headers/footers, binding gutter)
      -- as close parallel lines made sketchy by the rough painter may cross each other
      -- and create a mess for filling the area.
      -- So let's go for a solid fill to be safer.
      fillStyle = "solid"
  })
  return path
end

local framesetAdapter = require("resilient.adapters.frameset")

--- Draw a layout graph in the document.
--
-- @tparam SILE.types.measurement W Width of the drawing area
-- @tparam SILE.types.measurement H Height of the drawing area
-- @tparam table options Drawing options
function layout:draw (W, H, options)
  local ratio = SU.cast("number", options.ratio or 6.5)
  local rough = SU.boolean(options.rough, false)

  local oddFrameset, _ = self:frameset()
  -- Add fake page frame
  oddFrameset.page = {
    left = 0,
    top = 0,
    right = W,
    bottom = H,
  }
  local adapter = framesetAdapter(oddFrameset)
  local frames = adapter:solve()

  SILE.typesetter:pushHbox({
    width = W / ratio,
    height = H / ratio,
    depth = SILE.types.length(),
    outputYourself = function(node, typesetter, line)
      local saveX = typesetter.frame.state.cursorX
      local saveY = typesetter.frame.state.cursorY
      -- Scale to line to take into account stretch/shrinkability
      local outputWidth = node:scaledWidth(line)
      -- Force advancing to get the new cursor position
      typesetter.frame:advanceWritingDirection(outputWidth)
      local newX = typesetter.frame.state.cursorX
      -- Compute the target width, height, depth for the box
      local w = (newX - saveX):tonumber()
      local h = node.height:tonumber()
      -- Compute the page scaling ratio
      local wratio = w / frames.page:width():tonumber()
      local hratio = h / frames.page:height():tonumber()

      -- Compute and draw the PDF graphics path
      local painter = PathRenderer(rough and RoughPainter())
      local path
      path = buildFrameRect (painter, frames.bindinggutter, wratio, hratio, {
        stroke = "0.1pt",
        strokecolor = "225",
        fillcolor = "220"
      })
      SILE.outputter:drawSVG(path, saveX, saveY, w, h, 1)
      path = buildFrameRect (painter, frames.page, wratio, hratio, {
        stroke = "0.5pt",
        strokecolor = "black"
      })
      SILE.outputter:drawSVG(path, saveX, saveY, w, h, 1)
      path = buildFrameRect(painter, frames.textblock, wratio, hratio, {
        strokecolor = "black",
        fillcolor = "200"
      })
      SILE.outputter:drawSVG(path, saveX, saveY, w, h, 1)
      path = buildFrameRect(painter, frames.header, wratio, hratio, {
        strokecolor = "175"
      })
      SILE.outputter:drawSVG(path, saveX, saveY, w, h, 1)
      path = buildFrameRect(painter, frames.footer, wratio, hratio, {
        strokecolor = "175"
      })
      SILE.outputter:drawSVG(path, saveX, saveY, w, h, 1)
      path = buildFrameRect(painter, frames.margins, wratio, hratio, {
        strokecolor = "220"
      })
      SILE.outputter:drawSVG(path, saveX, saveY, w, h, 1)
    end
  })
end

return layout
