import QtQuick
import qs.Common
import qs.Widgets
import qs.Modules.Settings.Widgets

Item {
    id: root

    readonly property var timeoutOptions: [
        {
            text: I18n.tr("Never"),
            value: 0
        },
        {
            text: I18n.tr("1 second"),
            value: 1000
        },
        {
            text: I18n.tr("3 seconds"),
            value: 3000
        },
        {
            text: I18n.tr("5 seconds"),
            value: 5000
        },
        {
            text: I18n.tr("8 seconds"),
            value: 8000
        },
        {
            text: I18n.tr("10 seconds"),
            value: 10000
        },
        {
            text: I18n.tr("15 seconds"),
            value: 15000
        },
        {
            text: I18n.tr("30 seconds"),
            value: 30000
        },
        {
            text: I18n.tr("1 minute"),
            value: 60000
        },
        {
            text: I18n.tr("2 minutes"),
            value: 120000
        },
        {
            text: I18n.tr("5 minutes"),
            value: 300000
        },
        {
            text: I18n.tr("10 minutes"),
            value: 600000
        }
    ]

    function getTimeoutText(value) {
        if (value === undefined || value === null || isNaN(value))
            return I18n.tr("5 seconds");
        for (let i = 0; i < timeoutOptions.length; i++) {
            if (timeoutOptions[i].value === value)
                return timeoutOptions[i].text;
        }
        if (value === 0)
            return I18n.tr("Never");
        if (value < 1000)
            return value + "ms";
        if (value < 60000)
            return Math.round(value / 1000) + " " + I18n.tr("seconds");
        return Math.round(value / 60000) + " " + I18n.tr("minutes");
    }

    DankFlickable {
        anchors.fill: parent
        clip: true
        contentHeight: mainColumn.height + Theme.spacingXL
        contentWidth: width

        Column {
            id: mainColumn
            topPadding: 4
            width: Math.min(550, parent.width - Theme.spacingL * 2)
            anchors.horizontalCenter: parent.horizontalCenter
            spacing: Theme.spacingXL

            SettingsCard {
                width: parent.width
                iconName: "notifications"
                title: I18n.tr("Notification Popups")
                settingKey: "notificationPopups"

                SettingsDropdownRow {
                    settingKey: "notificationPopupPosition"
                    tags: ["notification", "popup", "position", "screen", "location"]
                    text: I18n.tr("Popup Position")
                    description: I18n.tr("Choose where notification popups appear on screen")
                    currentValue: {
                        if (SettingsData.notificationPopupPosition === -1)
                            return "Top Center";
                        switch (SettingsData.notificationPopupPosition) {
                        case SettingsData.Position.Top:
                            return "Top Right";
                        case SettingsData.Position.Bottom:
                            return "Bottom Left";
                        case SettingsData.Position.Left:
                            return "Top Left";
                        case SettingsData.Position.Right:
                            return "Bottom Right";
                        default:
                            return "Top Right";
                        }
                    }
                    options: ["Top Right", "Top Left", "Top Center", "Bottom Right", "Bottom Left"]
                    onValueChanged: value => {
                        switch (value) {
                        case "Top Right":
                            SettingsData.set("notificationPopupPosition", SettingsData.Position.Top);
                            break;
                        case "Top Left":
                            SettingsData.set("notificationPopupPosition", SettingsData.Position.Left);
                            break;
                        case "Top Center":
                            SettingsData.set("notificationPopupPosition", -1);
                            break;
                        case "Bottom Right":
                            SettingsData.set("notificationPopupPosition", SettingsData.Position.Right);
                            break;
                        case "Bottom Left":
                            SettingsData.set("notificationPopupPosition", SettingsData.Position.Bottom);
                            break;
                        }
                        SettingsData.sendTestNotifications();
                    }
                }

                SettingsToggleRow {
                    settingKey: "notificationOverlayEnabled"
                    tags: ["notification", "overlay", "fullscreen", "priority"]
                    text: I18n.tr("Notification Overlay")
                    description: I18n.tr("Display all priorities over fullscreen apps")
                    checked: SettingsData.notificationOverlayEnabled
                    onToggled: checked => SettingsData.set("notificationOverlayEnabled", checked)
                }

                SettingsToggleRow {
                    settingKey: "notificationCompactMode"
                    tags: ["notification", "compact", "size", "display", "mode"]
                    text: I18n.tr("Compact")
                    description: I18n.tr("Use smaller notification cards")
                    checked: SettingsData.notificationCompactMode
                    onToggled: checked => SettingsData.set("notificationCompactMode", checked)
                }
            }

            SettingsCard {
                width: parent.width
                iconName: "notifications_off"
                title: I18n.tr("Do Not Disturb")
                settingKey: "doNotDisturb"

                SettingsToggleRow {
                    settingKey: "doNotDisturb"
                    tags: ["notification", "dnd", "mute", "silent", "suppress"]
                    text: I18n.tr("Enable Do Not Disturb")
                    description: I18n.tr("Suppress notification popups while enabled")
                    checked: SessionData.doNotDisturb
                    onToggled: checked => SessionData.setDoNotDisturb(checked)
                }
            }

            SettingsCard {
                width: parent.width
                iconName: "lock"
                title: I18n.tr("Lock Screen", "lock screen notifications settings card")
                settingKey: "lockScreenNotifications"

                SettingsDropdownRow {
                    settingKey: "lockScreenNotificationMode"
                    tags: ["lock", "screen", "notification", "notifications", "privacy"]
                    text: I18n.tr("Notification Display", "lock screen notification privacy setting")
                    description: I18n.tr("Control what notification information is shown on the lock screen", "lock screen notification privacy setting")
                    options: [I18n.tr("Disabled", "lock screen notification mode option"), I18n.tr("Count Only", "lock screen notification mode option"), I18n.tr("App Names", "lock screen notification mode option"), I18n.tr("Full Content", "lock screen notification mode option")]
                    currentValue: options[SettingsData.lockScreenNotificationMode] || options[0]
                    onValueChanged: value => {
                        const idx = options.indexOf(value);
                        if (idx >= 0) {
                            SettingsData.set("lockScreenNotificationMode", idx);
                        }
                    }
                }
            }

            SettingsCard {
                width: parent.width
                iconName: "timer"
                title: I18n.tr("Notification Timeouts")
                settingKey: "notificationTimeouts"

                SettingsDropdownRow {
                    settingKey: "notificationTimeoutLow"
                    tags: ["notification", "timeout", "low", "priority", "duration"]
                    text: I18n.tr("Low Priority")
                    description: I18n.tr("Timeout for low priority notifications")
                    currentValue: root.getTimeoutText(SettingsData.notificationTimeoutLow)
                    options: root.timeoutOptions.map(opt => opt.text)
                    onValueChanged: value => {
                        for (let i = 0; i < root.timeoutOptions.length; i++) {
                            if (root.timeoutOptions[i].text === value) {
                                SettingsData.set("notificationTimeoutLow", root.timeoutOptions[i].value);
                                break;
                            }
                        }
                    }
                }

                SettingsDropdownRow {
                    settingKey: "notificationTimeoutNormal"
                    tags: ["notification", "timeout", "normal", "priority", "duration"]
                    text: I18n.tr("Normal Priority")
                    description: I18n.tr("Timeout for normal priority notifications")
                    currentValue: root.getTimeoutText(SettingsData.notificationTimeoutNormal)
                    options: root.timeoutOptions.map(opt => opt.text)
                    onValueChanged: value => {
                        for (let i = 0; i < root.timeoutOptions.length; i++) {
                            if (root.timeoutOptions[i].text === value) {
                                SettingsData.set("notificationTimeoutNormal", root.timeoutOptions[i].value);
                                break;
                            }
                        }
                    }
                }

                SettingsDropdownRow {
                    settingKey: "notificationTimeoutCritical"
                    tags: ["notification", "timeout", "critical", "priority", "duration"]
                    text: I18n.tr("Critical Priority")
                    description: I18n.tr("Timeout for critical priority notifications")
                    currentValue: root.getTimeoutText(SettingsData.notificationTimeoutCritical)
                    options: root.timeoutOptions.map(opt => opt.text)
                    onValueChanged: value => {
                        for (let i = 0; i < root.timeoutOptions.length; i++) {
                            if (root.timeoutOptions[i].text === value) {
                                SettingsData.set("notificationTimeoutCritical", root.timeoutOptions[i].value);
                                break;
                            }
                        }
                    }
                }
            }

            SettingsCard {
                width: parent.width
                iconName: "history"
                title: I18n.tr("History Settings")
                settingKey: "notificationHistory"

                SettingsToggleRow {
                    settingKey: "notificationHistoryEnabled"
                    tags: ["notification", "history", "enable", "disable", "save"]
                    text: I18n.tr("Enable History", "notification history toggle label")
                    description: I18n.tr("Save dismissed notifications to history", "notification history toggle description")
                    checked: SettingsData.notificationHistoryEnabled
                    onToggled: checked => SettingsData.set("notificationHistoryEnabled", checked)
                }

                SettingsSliderRow {
                    settingKey: "notificationHistoryMaxCount"
                    tags: ["notification", "history", "max", "count", "limit"]
                    text: I18n.tr("Maximum History")
                    description: I18n.tr("Maximum number of notifications to keep", "notification history limit")
                    value: SettingsData.notificationHistoryMaxCount
                    minimum: 10
                    maximum: 200
                    step: 10
                    unit: ""
                    defaultValue: 50
                    onSliderValueChanged: newValue => SettingsData.set("notificationHistoryMaxCount", newValue)
                }

                SettingsDropdownRow {
                    settingKey: "notificationHistoryMaxAgeDays"
                    tags: ["notification", "history", "max", "age", "days", "retention"]
                    text: I18n.tr("History Retention", "notification history retention settings label")
                    description: I18n.tr("Auto-delete notifications older than this", "notification history setting")
                    currentValue: {
                        switch (SettingsData.notificationHistoryMaxAgeDays) {
                        case 0:
                            return I18n.tr("Forever", "notification history retention option");
                        case 1:
                            return I18n.tr("1 day", "notification history retention option");
                        case 3:
                            return I18n.tr("3 days", "notification history retention option");
                        case 7:
                            return I18n.tr("7 days", "notification history retention option");
                        case 14:
                            return I18n.tr("14 days", "notification history retention option");
                        case 30:
                            return I18n.tr("30 days", "notification history retention option");
                        default:
                            return SettingsData.notificationHistoryMaxAgeDays + " " + I18n.tr("days");
                        }
                    }
                    options: [I18n.tr("Forever", "notification history retention option"), I18n.tr("1 day", "notification history retention option"), I18n.tr("3 days", "notification history retention option"), I18n.tr("7 days", "notification history retention option"), I18n.tr("14 days", "notification history retention option"), I18n.tr("30 days", "notification history retention option")]
                    onValueChanged: value => {
                        let days = 7;
                        if (value === I18n.tr("Forever", "notification history retention option"))
                            days = 0;
                        else if (value === I18n.tr("1 day", "notification history retention option"))
                            days = 1;
                        else if (value === I18n.tr("3 days", "notification history retention option"))
                            days = 3;
                        else if (value === I18n.tr("7 days", "notification history retention option"))
                            days = 7;
                        else if (value === I18n.tr("14 days", "notification history retention option"))
                            days = 14;
                        else if (value === I18n.tr("30 days", "notification history retention option"))
                            days = 30;
                        SettingsData.set("notificationHistoryMaxAgeDays", days);
                    }
                }

                SettingsToggleRow {
                    settingKey: "notificationHistorySaveLow"
                    tags: ["notification", "history", "save", "low", "priority"]
                    text: I18n.tr("Low Priority")
                    description: I18n.tr("Save low priority notifications to history", "notification history setting")
                    checked: SettingsData.notificationHistorySaveLow
                    onToggled: checked => SettingsData.set("notificationHistorySaveLow", checked)
                }

                SettingsToggleRow {
                    settingKey: "notificationHistorySaveNormal"
                    tags: ["notification", "history", "save", "normal", "priority"]
                    text: I18n.tr("Normal Priority")
                    description: I18n.tr("Save normal priority notifications to history", "notification history setting")
                    checked: SettingsData.notificationHistorySaveNormal
                    onToggled: checked => SettingsData.set("notificationHistorySaveNormal", checked)
                }

                SettingsToggleRow {
                    settingKey: "notificationHistorySaveCritical"
                    tags: ["notification", "history", "save", "critical", "priority"]
                    text: I18n.tr("Critical Priority")
                    description: I18n.tr("Save critical priority notifications to history", "notification history setting")
                    checked: SettingsData.notificationHistorySaveCritical
                    onToggled: checked => SettingsData.set("notificationHistorySaveCritical", checked)
                }
            }
        }
    }
}
